#ifndef PAPYTELEMAPP_H_INCLUDED
#define PAPYTELEMAPP_H_INCLUDED
/*
 * Papyrus Racing Games
 *
 * Date			Who		What
 * 05-apr-2002	RJC 	Created
 *
 ***************************************************************************
 *	
 *	Terminology:
 *		"The App" - refers to the telemetry gathering application
 *		"The Sim" - refers to the Papyrus racing simulator
 *		"Telemetry" - refers to any data sent to the app by the sim
 *		"State data" - refers to a specific type of telemetry that
 *			identifies changes in the sim's state (like the game
 *			has paused or unpaused, or a track has been loaded or
 *			unloaded).
 *		"Sample data" - refers to telemetry data that is generated
 *			each evenly-spaced time period (like the car's position
 *			and velocity, or the player's control inputs)
 *		"A sample" - refers to a set of sample data that was all
 *			generated by the sim at the same point in time in the
 *			simulated world.
 *	
 *	Overview:
 *	---------
 *	
 *	An app can tie into a Papyrus racing sim and gather telemetry
 *	from the sim as the sim is running.
 *	
 *	When the sim starts up, it looks to see whether an external
 *	telemetry application is running.  If so, the sim will
 *	establish communications with the app, and write telemetry
 *	(sample and state data) to the app as needed.
 *	
 *	To gather telemetry, then, the app must start up and identify
 *	itself as a sim-telemetry-gatherer before the sim is started.
 *	Only one such app can be active at a time.  If a second app
 *	tries to register itself as a gatherer, the attempt to register
 *	will fail.  More specifically, only one app can gather telemetry
 *	from any one instance of the sim.  If you're running multiple
 *	instances of the sim (on a server farm, for example), you can
 *	have one telemetry gathering app attached to each sim instance.
 *	See the "More detail" section below.
 *	
 *	Once the app has begun, it can make certain requests to the sim
 *	(like indicating which types of telemetry it is interested in).
 *	
 *	Once the sim has begun, it will process any such requests from
 *	the app.  The sim will also write both state data and sample data
 *	to the app as the data becomes available.  Examples of state
 *	data would be notifications whenever a track is loaded or
 *	unloaded, or whenever the player pauses or unpauses the game.
 *	Examples of sample data would be the chassis position and
 *	orientation, and the player input (throttle, brake, etc.).
 *	
 *	The app must be capable of reading the telemetry from the sim
 *	at least as quickly as the sim generates it.  As of this writing,
 *	if all telemetry is turned on the sim will be spitting data
 *	at the app at close to 300Hz, and will generate on the order of
 *	100Kbytes per second of data.  Typically, though, it will only
 *	spit data at 36Hz, generating on the order of a few Kbytes per
 *	second.
 *	
 *	The buffer into which the telemetry is written is of a limited
 *	size, so the app will need to quickly process the data to keep
 *	the buffer from overflowing. If the buffer does overflow,
 *	nothing catastrophic will happen.  The sim will continue to run,
 *	but it will stop writing data until space becomes available in
 *	the buffer, and so some telemetry will be lost.  The app will
 *	be able to query whether the buffer has overflowed.
 *	
 *	
 *	More detail:
 *	------------
 *	
 *	All of the data types, functions, etc. in the Papyrus telemetry
 *	API are in the namespace PapyTelem.  As such, you'll need to
 *	either qualify your use of the API components, or issue a
 *		using namespace PapyTelem;
 *	directive.
 *	
 *	The app indicates its desire to be a telemetry gatherer by
 *	calling the AppBegin() function, specifying the name of the
 *	simulator to which it wants to attach (e.g. "NR2003").  If
 *	this function succeeds, it will return true, and the app
 *	is then obliged to call AppEnd() before exiting.  The app
 *	should not call AppEnd() if it did not successfully call
 *	AppBegin().
 *	
 *	The system works by using a shared memory area that both
 *	the app and the sim have access to.  AppBegin() is the
 *	routine that creates this shared memory area.  When the
 *	sim starts up, it executes its corresponding SimBegin()
 *	routine, which looks to see if this shared memory area
 *	exists.  Requests made by the app are written into a
 *	queue in one part of this shared memory, and telemetry
 *	generated by the sim is written into a queue in another
 *	part of this memory.
 *	
 *	If multiple instances of the sim are to be run on one
 *	machine at the same time, then each instance that is to
 *	generate telemetry must be given an ID that is distinct
 *	from the others (this ID is used as part of the name given
 *	to the shared memory area).  This is done by setting the
 *		[Telemetry]
 *		id=x
 *	value in core.ini (or, equivalently, passing the sim a
 *	command line argument of the form
 *		-ko"0:Telemetry:id:x"
 *	where x is either...
 *		@	in which case the sim's server network port number
 *			will be used (32766 by default - see the
 *			[Communications] net_server_port setting in core.ini)
 *	or...
 *		some string
 *	The app would then need to pass this string (or port number,
 *	as a string) to AppBegin() so that it would connect to the
 *	proper instance of the sim.
 *	
 *	Once begun, the app can use the AppRequestDataItems() function
 *	to indicate to the sim what sample and/or state data it is
 *	interested in receiving.  It does not have to wait for the
 *	sim to start up before issuing this request.
 *	
 *	The simulator program updates its world at 36Hz.  This rate
 *	is fine for most things, but it isn't nearly sufficient for
 *	running the physics simulation.  To achieve the desired
 *	fidelity in the physics simulation, the simulator runs the
 *	physics a number of times each 36th of a second.  As of this
 *	writing, it runs the physics 8 times each 36th of a second,
 *	resulting in an underlying physics rate of 288Hz.  By default,
 *	the telemetry system will sample the physics at the 36Hz world
 *	update rate.  The app can use the AppRequestDataAtPhysicsRate()
 *	function to request data at the full 288Hz rate.  This will,
 *	obviously, generate 8 times the amount of data.
 *	
 *	By using AppRequestDataItems(), the app can ask for any of the
 *	data types listed in the eSimDataType enum, except those
 *	specifically noted.  The item types listed before
 *	kNoStateInfo are considered "sample" items.  Those that
 *	have been requested are generated as a group each 36th
 *	(or 288th) of a second.  While the system does not force
 *	you to do so, you should always request kSampleHeader if
 *	you also request any other sample item.  Items below
 *	kNoStateInfo are considered "state" items.  Those that have
 *	been requested are generated by the sim as needed.
 *
 *	Player car gauge data and opponent car data will only be
 *	sampled at 36Hz even when 288Hz is the selected sample rate.
 *	The opponent car data will be sampled even when the player
 *	car is not on track. So your app must be able to cope with
 *	sample data that does not contain all the fields you requested
 *	in every tick.
 *	
 *	The heart of the system is the AppWaitForNewSample()
 *	function.  Calling AppWaitForNewSample() will
 *	read telemetry items from the sim until either a complete
 *	sample has been read, or a state item has been read, or
 *	the specified timeout has expired (approximate - do NOT
 *	use the timeout facility of this function to try to get
 *	your app to do something on an even time schedule).
 *	Specifying a zero timeout will cause AppWaitForNewSample
 *	to return immediately if neither a sample nor a bit of
 *	state telemetry is available.  If a complete sample has
 *	been read, AppWaitForNewSample will return true.
 *
 *	The telemetry system has buffers that will hold one of
 *	each type of telemetry item.  When AppWaitForNewSample
 *	reads a telemetry item from the sim, it stores it in the
 *	buffer for that item type, and sets an internal flag
 *	indicating that the app can query that item.  The app
 *	can call AppGetSimData, specifying the item type,
 *	to get at the data.  It will return a pointer to the data
 *	item if it is available, or NULL otherwise.
 *	
 *	Any of these item buffers could be overwritten the
 *	next time the app calls AppWaitForNewSample, so the app
 *	should NOT maintain pointers obtained from AppGetSimData
 *	across calls to AppWaitForNewSample.  The app may
 *	explicitly clear a particular item type by calling
 *	AppClearSimData for that type, or it can clear all sample
 *	items by calling AppClearSample
 *
 *	When AppWaitForNewSample returns true, the app can then call
 *	AppGetSimData() repeatedly, specifying the eSimDataType items
 *	it has requested, to access the sample data.  Once the app
 *	is done with this sample, it should call AppClearSample()
 *	to ensure that it can not inadvertently re-read part of this
 *	sample again.
 *	
 *	Note that since calling AppRequestDataItems() is asynchronous
 *	to the sim's telemetry generation, the app cannot assume
 *	that the next sample it reads after having called this
 *	function will reflect this request.  Thus, the app should
 *	ALWAYS test that AppGetSimData() has returned a non-NULL
 *	pointer.
 *	
 *	When AppWaitForNewSample() returns false, it will either be
 *	because the timeout expired, or a piece of state data was
 *	read, in which case its newStateData output parameter will
 *	be set to the eSimDataType enum identifying the piece of
 *	state data that was read from the sim.  The app can then
 *	call AppGetSimData() with this type to see the new state data.
 *	Note that calling AppClearSample() clears only sample data,
 *	not state data.
 *	
 *	
 *	Important notes:
 *	----------------
 *	
 *	The sim can generate telemetry at a high rate.  The buffer
 *	into which the telemetry is written is not particularly
 *	large, so the app must be prepared to read the telemetry
 *	quickly.  If the app falls behind and the buffer overflows,
 *	the sim will have to discard some telemetry.  When this
 *	happens, it's possible that only part of a sample will be
 *	written (though the sim will never write only part of a
 *	specific data item).  The app must be prepared to deal
 *	with partial samples.
 *	
 *	By default, the sim runs as a HIGH_PRIORITY_CLASS process.
 *	The sim is multi-threaded, and it runs all of its real-time
 *	threads at at least THREAD_PRIORITY_ABOVE_NORMAL.  All of
 *	these real-time threads run on a time schedule and will
 *	cooperatively put themselves to sleep when their processing
 *	of the current game loop is complete.  The sim also has a
 *	non-real-time thread that it runs at THREAD_PRIORITY_BELOW_NORMAL.
 *	This thread never sleeps for more than a brief period, and will
 *	happily eat up almost all available CPU time.
 *	
 *	For your app to run effectively, you will either need to
 *	also run it as a HIGH_PRIORITY_CLASS process by calling
 *	SetPriorityClass(GetCurrentProcess(), HIGH_PRIORITY_CLASS);
 *	or by telling the sim to run at the default priority class
 *	by passing it
 *	-kh0
 *	on the command line (that's a zero, not an oh).
 *
 *
 *	Sending telemetry to the replay
 *	-------------------------------
 *
 *	This is not intended to be a treatise on the format or
 *	contents of the sim's .rpy file.  It is intended simply
 *	as a "heads up."
 *
 *	There are settings in the [Replay] section of the sim's
 *	core.ini configuration file that allow dumping telemetry
 *	data to the replay system.  Note that when these settings
 *	are used the sim acts as both the telemetry generator,
 *	and the telemetry gatherer (the "app").  Therefore, using
 *	an external telemetry gathering application and these
 *	settings at the same time is not possible.
 *	
 *	The settings, with their defaults, are as follows...
 *	
 *		[Replay]
 *		record_telem=0
 *		record_telem_at_physics_rate=0
 *		record_telem_sample=0
 *		record_telem_state_a=0
 *		record_telem_state_b=0
 *
 *	record_telem - when set to 1 causes the sim to attempt
 *		to act as the telemetry gatherer.  If the sim can
 *		successfully call AppBegin, it will record the
 *		indicated telemetry to the replay.
 *	record_telem_at_physics_rate - when set to 1, will
 *		record sample data at the 288Hz physics rate.
 *	record_telem_sample - Set this to the bitwise or
 *		of (1 << eSimDataType) for the desired sample items.
 *		For example, to record kSampleHeader and
 *		kChassisData, you'd set this value to 3.
 *	record_telem_state_a - Same as above, but subtract
 *		off (kNoStateInfo + 1) from the desired enum value.
 *		For example, to dump kCarSetup and kPitStop, you'd
 *		set this to 24, which is
 *		(1 << (kCarSetup - (kNoStateInfo + 1))) |
 *		(1 << (kPitStop - (kNoStateInfo + 1)))
 *	record_telem_state_b - Unused since there are not yet
 *		more than 32 types of state telemetry.
 *
 *	The current eSimDataType-to-replay item type mapping is...
 *
 *		kSampleHeader		= 100
 *		kChassisData		= 101
 *		kWheelData			= 102
 *		kTireData			= 103
 *		kDrivelineData		= 104
 *		kDriverInput		= 105
 *
 *		kGaugeData			= Not recorded
 *		kOpponentCarData	= Not recorded
 *
 *		kCurrentWeekend		= Not recorded
 *		kDriverInCar		= Not recorded
 *		kGameIsPaused		= Not recorded
 *		kCarSetup			= 116
 *		kPitStop			= 117
 *		kDriverEntry		= Not recorded
 *		kDriverWithdrawl	= Not recorded
 *		kSessionInfo		= Not recorded
 *		kLapCrossing		= Not recorded
 *		kStandings			= Not recorded
 *
 *	Note that not all state items are recorded.  In some
 *	cases, they simply are not necessary, and in others
 *	the sim's replay system is not "recording" when the
 *	telemetry is generated.
 *
 *	For each telemetry item type that is recorded to the
 *	replay, the item's struct is written directly to the
 *	replay tape.  The data is not encoded or compressed.
 *	If record_telem_at_physics_rate is set, 8 instances
 *	of each requested sample item will appear in a
 *	given frame.  Since the data is not compressed, it
 *	will use a lot of replay memory.  In some scenarios,
 *	it will use far more replay memory than everything else
 *	combined.
 *	
 ***************************************************************************
 */


/*
 * Include files
 */

// It is the intent that this file be distributed with the game
// for public use, so do NOT include any Papyrus specific header
// files here!


namespace PapyTelem
{
// The Microsoft Visual Studio compiler directive to cause
// structures to be packed to 1 byte alignment.  For example,
// struct { char a, float b}; would take 5 bytes, and an
// array of two of these would take 10 bytes.
#pragma pack (push, 1)

// The types of telemetry that the app can ask the sim to generate.
// The first group are sample items, and will be generated as a
// group each sampling period (36 or 288Hz).
// The second group are sample items, but will only be generated
// at a 36Hz period.
// The third group are state items, and will be generated as the
// need arises.
// You should never request either kNoStateInfo, or kSampleFooter,
// as they are used internally by the telemetry system.
enum eSimDataType {
	kSampleHeader,
	kChassisData,
	kWheelData,
	kTireData,
	kDrivelineData,
	kDriverInput,

	// these samples are available at only 36Hz, even when 288Hz is selected
	kGaugeData = 64,
	kOpponentCarData,

	// Add any new per-sample items above here.
		// The app must NOT use this data type
		kNoStateInfo = 127,
	// Add any "state" data below here.

	kCurrentWeekend,
	kDriverInCar,
	kGameIsPaused,
	kCarSetup,
	kPitStop,
	kDriverEntry,
	kDriverWithdrawl,
	kSessionInfo,
	kLapCrossing,
	kStandings,


	// DO NOT ADD ANY ITEMS BELOW HERE
	// The app must NOT use this data type.
	kSampleFooter	= 255
};


////////////////////////////////////////////////////////////////////////////
// The following data are sample items.
////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------
// Each time the driver gets into the car (including whenever they hit
// shift-R to reset the car), the sequence number will change.
// Each 36th of a second when the player physics is run, tick will
// increment by one.
// If you're requesting data at the physics rate, you'll get 8 samples
// each 36th of a second with the same tick value, but the step value
// will count from [0..7].  If you're sampling at 36Hz, step will always
// be 7.
// time counts the number of simulation seconds since sequence was
// incremented.  It does not increase while the game is paused.
typedef struct {
	unsigned int	sequence;		// Increments each time physics is reset
	unsigned int	tick;			// Increments each 1/36th of a second
	int				step;			// From 0..N if AtPhysicsRate()
	double			time;			// Elapsed sim-world seconds since reset
} SampleHeader;


//------------------------------------------------------------------------
// World coordinates:
//		- right handed, with +z = up
// From the driver's point of view sitting at the center of gravity of the car,..
// Car coordinates:
//		- right handed, with +z = up, +x = towards nose, +y towards left side
//		- +yaw rotates the +x axis into the +y axis
//		- +pitch rotates the +z axis into the +x axis
//		- +roll rotates the +y axis into the +z axis
typedef struct {
	float			trkPct;			// Percentage of way around the track
	float			r[3];			// position of car c.o.g. [x,y,z] (meters)
	float			v[3];			// velocity [x,y,z] (meters/sec)
	float			q[3];			// orientation [yaw,pitch,roll] (radians)
	float			w[3];			// angular velocity [yaw,pitch,roll] (radians/sec)
	float			N[3];			// unit normal to track surface [x,y,z]
	float			H[2];			// unit track heading vector [x,y]
	float			steerT;			// torque on steering shaft (N*m, +=clockwise)
} ChassisData;

// NOTE:
// For the chassis orientation, yaw is applied first, then pitch, then roll.
// You can generate a 3x3 rotation matrix from car coordinates to world
// coordinates from ChassisData.q as follows...
//		- Let the prefix 'c' mean cos(x), and the prefix 's' mean sin(x)
//		- Let the suffix 'y' mean yaw, 'p' mean pitch, and 'r' mean roll
// so that, for example, 'cp' means cos(pitch), or cos(ChassisData.q[1]).
// Then...
//		m[0][0] = cp*cy;
//		m[0][1] = cy*sp*sr - cr*sy;
//		m[0][2] = cr*cy*sp + sr*sy;
//		m[1][0] = cp*sy;
//		m[1][1] = cr*cy + sp*sr*sy;
//		m[1][2] = cr*sp*sy - cy*sr;
//		m[2][0] = -sp;
//		m[2][1] = cp*sr;
//		m[2][2] = cp*cr;
// Multiplying a vector by this matrix will rotate it from the car's
// coordinate system into the world coordinate system.  Multiplying
// a vector by the transpose of this matrix goes the other way.
//
// See the sample program for an example.  It computes lateral,
// longitudinal, and normal G loads from (sequences of) this data.



//------------------------------------------------------------------------
// Some of the items below have arrays of information per wheel.
// These arrays are indexed in this order.
enum {
	kWheelLeftFront,
	kWheelRightFront,
	kWheelLeftRear,
	kWheelRightRear
};

// Conversion factor: kiloPascal to pounds per square inch
const float kPa_to_PSI = 0.1450377f;
typedef struct {
	struct {
		float			v;			// rot. velocity (radians/sec)
		float			defl;		// shock compression (meters)
		float			camber;		// wheel relative to road (radians)
		float			Fz;			// force normal to road (N)
	} w[4];							// LF, RF, LR, RR
} WheelData;

//------------------------------------------------------------------------
typedef struct {
	struct {
		float			temp[3];	// Inner, Middle, Outer, (degrees C)
		float			pressure;	// Hot pressure (gauge) (kPa (kiloPascal))
	} w[4];							// LF, RF, LR, RR
} TireData;

//------------------------------------------------------------------------
typedef struct {
	float				engRPM;		// engine RPM
	float				engPower;	// engine power output (N*m/sec)
	float				engT;		// torque from engine (N*m)
} DrivelineData;

//------------------------------------------------------------------------
typedef struct {
	float				throttle;	// Percent applied
	float				brake;		// Percent applied
	float				clutch;		// Percent released (1 => foot off clutch)
	float				steer;		// radians at lf wheel (no ackerman)
	int					gear;		// reverse = <0, neutral = 0, fwd gear = >0
} DriverInput;

//------------------------------------------------------------------------
typedef struct {
	float				rpm;		// engine RPM
	float				waterTemp;	// in degrees Celcius
	float				fuelPress;	// in bar (1 bar ~= 14.7 psi)
	float				oilTemp;	// in degrees Celcius
	float				oilPress;	// in bar
	float				voltage;	// electrical system voltage
	bool				rpmWarning			: 1;	// RPM warning light
	bool				waterTempWarning	: 1;	// water temp warning light
	bool				fuelPressureWarning	: 1;	// fuel pressure warning light
	bool				oilPressureWarning	: 1;	// oil pressure warning light
} GaugeData;

//------------------------------------------------------------------------
// See the comments for ChassisData for details of these coordinate values.
// Telemetry of this type is for all (available) cars except the local
// player's car. That is, this telemetry will include the pace car,
// AI cars, and remote player cars.  The pointer returned by
// AppGetSimData(kOpponentCarData) will be to an array of these structs.
// Use (AppGetSimDataSize(kOpponentCarData)/sizeof(OpponentCarData)) to
// determine the number of valid elements in the array.
// If you are collecting this telemetry from a client in a multiplayer
// race, you should be aware that, due to bandwidth limitations between
// the client and the server, you are likely to only receive this telemetry
// for a subset of the cars participating in the race.
typedef struct {
	int					carIdx;
	float				trkPct;		// Percentage of way around the track
	float				r[3];		// position of car c.o.g. [x,y,z] (meters)
	float				v[3];		// velocity [x,y,z] (meters/sec)
	float				q[3];		// orientation [yaw,pitch,roll] (radians)
} OpponentCarData;

////////////////////////////////////////////////////////////////////////////
// The following data are state items.
// They are generated at the first opportunity after
// the event occurs.  This will generally be immediate unless
// there is a data overrun in the sim telemetry buffer.
////////////////////////////////////////////////////////////////////////////

//------------------------------------------------------------------------
// You get one of these whenever a new set of race weekend parameters
// is proposed, or when the weekend becomes active or inactive.
// See the game's readme.txt for a description of the .sessions
// and .options strings.  They're described in the off-line league
// section.
typedef struct {
	bool				atTrack;	// Does the sim have a track loaded?
	char				track[64];	// track directory
	float				trackLength;// meters
	char				sessions[64];// See off-line league section of
	char				options[32];// readme.txt for descriptions.
} CurrentWeekend;

//------------------------------------------------------------------------
// You will get one of these whenever the player gets into, or out of
// the car.  You might, or might not, get these messages when the
// player hits shift-R to reset the car (you get them in single player,
// you don't if the player is connected to a remote server in a
// multiplayer race).
typedef struct {
	bool				inCar;		// Is player in-car?
} DriverInCar;


//------------------------------------------------------------------------
// You get one of these whenever the game is paused or unpaused,
// even if the player is not currently driving.
typedef struct {
	bool				paused;		// Is the game paused?
} GameIsPaused;


//------------------------------------------------------------------------
// You get one of these each time the player gets into the car.
// It represents the setup that was applied to the car.
// Note that this structure is specific to the Papyrus NASCAR
// Racing series, and its contents would likely be different
// for simulations of other racing series.
typedef struct {
	struct {
		float			pressure;	// (cold, gauge) tire pressure (PSI)
		float			spring;		// spring rate (not wheel rate) (lbs/inch)
		float			camber;		// degrees - negative means top tilted in
		int				bump;		// shock compression setting - unitless
		int				rebound;	// shock rebound setting - unitless
		float			rideHeight;	// frame rail height (inches)
		float			weight;		// static corner weight (lbs)
	} corner[4];					// LF, RF, LR, RR
	float				wedge;		// cross weight (lbs - positive = more RF<->LR)
	struct {
		float			barDiam;	// anti-roll bar diameter (inches)
		float			toeOut;		// toe out (inches)
		float			caster[2];	// degrees LF, RF
		float			grilleTape;	// percentage of air inlet area taped up
	} front;
	struct {
		float			barDiam;	// anti-roll bar diameter (inches)
		float			toeOut;		// toe out (inches)
		float			trackBar[2];// track bar height (inches LR, RR)
		float			spoiler;	// degrees
	} rear;
	float				gearRatio[6];// gear ratio 1st through 6th
	float				diffRatio;	// differential ratio
	float				brakeBias;	// percentage braking at the front
	float				steerRatio;	// steering rack ratio
	float				fuel;		// gallons
} CarSetup;


//------------------------------------------------------------------------
// You will get one of these each time the player pits, and
// the values will match the new state after adjustments were
// made during the pit stop.  You might consider calling
// AppClearSimData(kPitStop) each time you receive a kCarSetup
// state item, to reduce possible confusion as to what setup is
// on the car.
// Note that this structure is specific to the Papyrus NASCAR
// Racing series, and its contents would likely be different
// for simulations of other racing series.
typedef struct {
	struct {
		bool			changed;	// Was this tire changed?
		float			pressure;	// (cold, gauge) tire pressure (PSI)
	} tire[4];						// LF, RF, LR, RR
	float				wedge;		// cross weight (lbs - positive = more RF<->LR)
	float				grilleTape;	// percentage of air inlet area taped up
	float				trackBar[2];// track bar height (inches LR, RR)
	float				fuel;		// gallons
} PitStop;


//------------------------------------------------------------------------
// You will get a DriverEntry each time a "driver" joins the race.
// Drivers can be either human players, or computer controlled
// cars (AI).  AI drivers are so flagged in the driverFlags field.
// The pace car is also registered as a driver, and is also flagged.
// Any other telemetry that refers to a specific car or driver
// will use an integer index that matches the carIdx field of
// the DriverEntry.
// carIdx's can range from 0..(kMaxCars-1), inclusive.
// kCarIdxInvalid is used as an 'invalid' car index.
enum {
	kMaxCars			= 48,
	kCarIdxInvalid		= -1
};
enum eDriverFlags {
	kDriverIsPaceCar	= 0x00000001,
	kDriverIsAI			= 0x00000002,
	kDriverIsBoss		= 0x00000004,
	kDriverIsAdmin		= 0x00000008
};
typedef struct {
	int					flags;		// bitwise or of eDriverFlags
	int					carIdx;		// index (used in lap crossings, ...)
	int					carMake;	// 
	char				carNum[4];	// Painted on car
	char				name[64];	// driver name
	char				carFile[32];// .car file name
} DriverEntry;


//------------------------------------------------------------------------
// You will get a DriverWithdrawl whenever a driver leaves the race.
// You will also get a flood of withdrawls, one for each possible
// carIdx, when the user first connects to a game server (even if
// they're just starting a testing session, since this involves a
// 'local server').  So you should not assume that if you get a
// withdrawl, that you will have previously received an entry.
typedef struct {
	int					carIdx;
} DriverWithdrawl;


//------------------------------------------------------------------------
// These are the different session types.  Warmup/Happy Hour will
// show up as a practice session.  NASCAR uses the stQualifyLone
// session type for qualifying.
enum eSessionType {
	kSessionTypeInvalid,
	kSessionTypeTesting,
	kSessionTypePractice,
	kSessionTypeQualifyLone,
	kSessionTypeQualifyOpen,
	kSessionTypeRace
};

// Any given session will proceed through one or more of these
// session states.  Not all session types use all the states.
// During ssGetInCar, the sim is waiting for drivers to get
// into their cars before proceeding.
// ssWarmup is during "Gentlemen, start your engines!"
// ssParadeLaps is used during the pace lap at the start of the race.
// ssRacing is used during the majority of most sessions.  It
// includes caution periods during a race.
// ssCheckered us used during the period of time from when
// the checkered flag is shown to the race leader until the last
// car running takes the checkered flag.
// ssCoolDown covers the period of time after the session is
// complete until the next session starts, or the weekend is
// ended.
enum eSessionState {
	kSessionStateInvalid,
	kSessionStateGetInCar,
	kSessionStateWarmup,
	kSessionStateParadeLaps,
	kSessionStateRacing,
	kSessionStateCheckered,
	kSessionStateCoolDown
};

// The current track status - is the yellow flag out?
enum eSessionFlag {
	kFlagGreen,
	kFlagYellow,
	kFlagRed
};

// You will get a SessionInfo periodically whenever a race
// weekend is active.
// - sessionNum is the session index (the first sessionNum is 0).
// - sessionCookie will change whenever a session is restarted.
// It's pretty important that you pay attention to these two.
// Whenever the sessionNum or sessionCookie changes, you'll want
// to clear out any information you may have cached for this
// and any later sessions (like lap crossings and standings -
// see below).
// - sessionType is one of the eSessionType values above.
// - sessionState is one of the eSessionState values above.
// - sessionFlag is one of the eSessionFlag values above.
// - currentTime is the current session time - the number of
// elapsed simulation seconds since the start of the session.
// Simulation time does not advance while the game is paused.
// Simulation time (generally) advances more quickly than
// real time when the sim is running in 'accelerated' time.
// - timeRemaining reflects what is shown on the "time remaining"
// field on the race weekend screen.  It is the number of seconds
// remaining in the current state of the current session.  It is
// possible that this value will be -1 immediately after a state
// transition
// - lapsRemaining will only be valid during sessions whose length
// is limited to a particular number of laps (qualifying and race).
typedef struct {
	int					sessionNum;
	int					sessionCookie;	// changes if session restarted
	int					sessionType;	// eSessionType
	int					sessionState;	// eSessionState
	int					sessionFlag;	// eSessionFlag
	float				currentTime;
	float				timeRemaining;
	int					lapsRemaining;
} SessionInfo;


//------------------------------------------------------------------------
// lapNum in LapCrossing may range from 0..(kMaxLaps-1), inclusive.
// If you intend to use the LapCrossing telemetry, you are quite
// likely to want to have a piece of state data such as this...
//		myCrossingEntry		lapTable[kMaxCars][kMaxLaps];
// You could use LapCrossing as myCrossingEntry, but the table would
// contain redundant information (carIdx would be stored in each entry).
enum {
	kMaxLaps				= 512
};

// Each lap crossing will have zero or more of these flag bits set.
// - timeInvalid will be flagged if there was some abnormal situation
// during this lap for the given driver, and so the lap time should not
// be considered 'valid'.  For example, if a driver cuts through the
// inner loop at Watkins Glen, they will gain an unfair advantage, and
// so the time for that lap is not meaningful.
// - offTrack will be flagged if the driver goes off track (onto the
// grass or into a sand trap, for example) during the lap.
// - pitted will be set if the driver stopped in their pit stall.
// - blackFlag will be set if the driver has a black flag pending
enum eLapCrossingFlags {
	kCrossTimeInvalid	= 0x00000001,
	kCrossOffTrack		= 0x00000002,
	kCrossPitted		= 0x00000004,
	kCrossBlackFlag		= 0x00000008
};

// Lap crossings are the basis for timing and scoring.  A lap
// crossing is generated for a particular car each time that car
// crosses the finish line.  If you collect and store all of the
// lap crossings by all the cars during a given session, you can
// answer most scoring related queries you might want to make.
// Like "what was each driver's fastest lap time?", and "which
// driver had the overall fastest lap, and what was that lap time?",
// and "who lead lap 42?", and "how many laps down was car index
// 12 on lap 193?" (some are easier to compute than others :-)
//
// LapCrossing telemetry will be generated as soon as practicable
// after the data becomes available.  If you connect as a client
// to a race server, the server will not send you lap crossings
// for sessions that have already been completed.  For example,
// if you connect to a race server during qualifying, the server
// will not send you all the lap crossings that occurred during
// practice.  It will send you all the lap crossings that have
// occurred so far during the currently active session (though
// it may take some time to do so).
// You are guaranteed to receive LapCrossing's for a particular
// car in the order in which they occurred.
// You are NOT guaranteed to receive all lap crossings in the
// order in which they occurred.  For example, if two cars
// cross the start finish line at nearly the same time, you
// will receive the lap crossing for the car with the smaller
// car index first, even if the car with the higher car index
// crossed before it.

// IMPORTANT:
//		- Whenever the SessionInfo telemetry's sessionNum or
//		sessionCookie value changes, you must clear any cache
//		of lap crossings you may have made.
//		- Since you must clear cached crossings whenever a
//		session change occurs, and since you only know about
//		such changes if you've requested SessionInfo telemetry,
//		IF YOU WANT TO RECEIVE LAP CROSSING TELEMETRY, YOU
//		MUST ALSO ASK TO RECEIVE SessionInfo TELEMETRY.
// - carIdx will be from 0..kMaxCars-1, and will match that
// given in the DriverEntry telemetry.
// - lapNum will count sequentially from 0 for each car.
// - flags will be the bitwise-or of eLapCrossingFlags.
// - crossedAt is the session time at which the crossing
// occurred.  You may compare this with crossedAt for any
// other LapCrossing in this session.
typedef struct {
	int					carIdx;
	int					lapNum;		// 0..(kMaxLaps-1)
	int					flags;		// bitwise or of eLapCrossingFlags
	double				crossedAt;	// Session time at lap crossing
} LapCrossing;


//------------------------------------------------------------------------
// The potential reasons a driver may be out the race.
enum eReasonsOut {
	kReasonOutNotOut,			// (default) not out of race
	kReasonOutBrakeFailure,
	kReasonOutCoolantLeak,
	kReasonOutRadiatorLeak,
	kReasonOutEngineFailure,
	kReasonOutEngineHeader,
	kReasonOutEngineValve,
	kReasonOutEnginePiston,
	kReasonOutEngineGearbox,
	kReasonOutEngineClutch,
	kReasonOutEngineCamShaft,
	kReasonOutEngineIgnition,
	kReasonOutEngineFire,
	kReasonOutEngineElectrical,
	kReasonOutFuelLeak,
	kReasonOutFuelInjector,
	kReasonOutFuelPump,
	kReasonOutFuelLine,
	kReasonOutOilLeak,
	kReasonOutOilLine,
	kReasonOutOilPump,
	kReasonOutOilPressure,
	kReasonOutSuspensionFailure,
	kReasonOutTirePuncture,
	kReasonOutTireProblem,
	kReasonOutWheelProblem,
	kReasonOutAccident,
	kReasonOutRetired,
	kReasonOutDisqualified,
	kReasonOutNoFuel,
	kReasonOutBrakeLine,
	kReasonOutLostConnection,
	kReasonOutEjected
};

// Info per-car.
typedef struct {
	int					carIdx;		// as in DriverEntry, or -1 if invalid
	float				time;		// behind leader in race, lap time otherwise
	int					lap;		// lap number
	int					lapsLead;	// if race session, number of laps lead
	int					reasonOut;	// eReasonOut
	int					incidents;	// number of 'incidents' for this driver.
} StandingsEntry;

enum eStandingsFlags {
	kStandingsOfficial	= 0x00000001,
	kStandingsForRace	= 0x00000002
};

// Standings telemetry will be generated periodically during the
// session.  When a session is complete and the results have been
// computed and made official, the standings will be sent one
// final time for this session, and will be flagged as 'official'.
//
// If you connect to a race server in the middle of a race weekend,
// the race server will send you the official results from all
// prior sessions in the race weekend, and telemetry for those
// sessions will be generated.
//
// If you cache these standings, you should take car to clear them
// when the sim leaves the current race weekend, and also when
// a session is restarted (see the SessionInfo telemetry).
// Clearly, then, it would be wise to request kCurrentWeekend
// and kSessionInfo telemetry if you request kStandings telemetry.
//
// Some of the fields are only valid during a race session,
// and are so marked below.  For convenience, the flags field
// will have the kStandingsForRace flag set if so.
//
// - fastestLap is filled in for the driver with the fastest
// lap in the session.  fastestLap.carIdx will be kCarIdxInvalid
// if there are no valid laps to be considered.
// - position is ordered with the leader at index 0, the second place
// car at index 1, etc.  If position[x].carIdx == kCarIdxInvalid,
// then this and all higher indexed positions are empty, and should
// be ignored.
typedef struct {
	int					flags;				// bitwise-or of eStandingsFlags
	int					sessionNum;
	float				averageLapTime;		// race session only
	int					lapsComplete;		// race session only
	int					numCautionFlags;	// race session only
	int					numCautionLaps;		// race session only
	int					numLeadChanges;		// race session only
	StandingsEntry		fastestLap;
	StandingsEntry		position[kMaxCars];
} Standings;



////////////////////////////////////////////////////////////////////////////

			// Initialize the application end of the telemetry system.
			// This must be done before the sim is started.
			// - simName should be the name of the sim to which you want
			// to connect.  This should be the first part of the
			// version number string shown on the sim's copyright screen.
			// For example, if the version string is "NR2003.1.0.0.0",
			// then you would pass "NR2003" as simName.
			// - Unless you are running multiple instances of the sim
			// on one machine at the same time, you can ignore the
			// id parameter.  See the discussion in the More Detail
			// section above.
			// AppBegin will return false if it did not succeed.
bool		AppBegin(const char *simName, const char *id = NULL);

			// Will return true if AppBegin was called successfully,
			// and AppEnd has not yet been called.
bool		AppBegunQ();

			// Will return true once the simulator has started and
			// initialized its end of the telemetry system.  It will
			// continue to return true until the sim exits, or
			// AppEnd is called.
bool		AppCheckIfSimActiveQ();

			// If you've successfully called AppBegin, you should
			// call AppEnd to clean up the application end of the
			// telemetry system before your program terminates.
			// It is not necessary for you to wait until the sim
			// exits to use AppEnd.  Note, however, that once
			// you've called AppEnd, you will not be able to
			// successfully call AppBegin again until the sim does
			// exit.
void		AppEnd();



			// The following three functions will write your request
			// into a buffer that the sim will read and process at
			// its convenience.  If the request will not fit in the
			// available space in the buffer, it will return false.

			// Pass true to receive samples at 288Hz, false for 36Hz.
			// By default, samples are generated at 36Hz.
bool		AppRequestDataAtPhysicsRate(bool yes);

			// The set you pass here replaces your prior request.
			// Note that you must also call AppEnableSampling()
			// By default, no data items are generated.
bool		AppRequestDataItems(int nItems, const eSimDataType *itemTypes);

			// Pass true to enable sample generation, false to disable.
			// This only affects sample items.  Requested state items
			// will still be generated even if this is called with false.
			// By default, telemetry generation is disabled.
bool		AppEnableSampling(bool yes);




			// This is the routine you call to read telemetry (both
			// sample data, and state date) from the sim.
			// Pass a negative timeout value to wait forever (not recommended)
			// Pass 0 to return immediately once a complete sample,
			// has been read, or a state item has been read, or the
			// telemetry buffer is empty.
			// There are three possibilities when this routine returns:
			//	+ it returns true
			//		- a new sample is available.  use appGetSimData() for
			//		  each type of sample data you've requested (being
			//		  careful to test for NULL).
			//		- newStateData will be set to kNoStateInfo,
			//	+ it returns false, and newStateData != kNoStateInfo
			//		- a new piece of state data is available.  call
			//		  appGetSimData(newStateData) to examine the state data.
			//	+ it returns false, and newStateData == kNoStateInfo
			//		- no new state info was read, and there is not yet a
			//		  new sample availble.  It is possible that part of
			//		  a new sample may have been read.
bool		AppWaitForNewSample(eSimDataType *newStateData, int timeoutMillisecs);

			// Clear the most recently generated sample.  It is not
			// necessary for you to call this, but you might use it to
			// prevent inadvertent reuse of a sample.
void		AppClearSample();

			// You can use this to clear a particular bit of telemetry.
void		AppClearSimData(eSimDataType dataType);

			// Access the most recently generated bit of telemetry
			// of the specified type.  See Overview discussion above
			// for importance of checking the return value for NULL.
const void	*AppGetSimData(eSimDataType dataType);

			// Determine the size of the most recently generated bit
			// of telemetry of this type.  For most items, this will
			// return either 0 if the item has been cleared, or the
			// sizeof() the item's struct.  If a particular item's
			// description indicates that the sim will generate multiple
			// copies of the item at once, then this will return
			// N * sizeof(the struct).  Currently (17-apr-2003), the
			// only such item is kOpponentCarData.
int			AppGetSimDataSize(eSimDataType dataType);



			// See if the telemetry buffer has overrun.  Each time the
			// sim tries to write a bit of telemetry, and there is not
			// enough space in the buffer for it, it will set this flag.
			// Once set, this flag will remain true until you (the app)
			// clears it.
bool		AppCheckIfSimDataOverrunQ();

			// Clear the telemetry overrun flag.  
void		AppClearSimDataOverrun();



			// AppRequestDataAtPhysicsRate, AppRequestDataItems, and
			// AppEnableSampling write your requests into a small buffer
			// that the sim reads and processes at its leisure.  If you
			// call one of these functions, and it returns false, it will
			// be because this buffer is currently full.  You can call
			// this function to flag that this has happened.  The sim
			// will clear this flag whenever it reads a request from
			// the buffer.
void		AppSetAppDataOverrun();

			// Will return true if you have called AppSetAppDataOverrun,
			// and the sim has not yet cleared this flag.
bool		AppCheckIfAppDataOverrunQ();


#pragma pack(pop)
}; // namespace PapyTelem

#endif	// #ifndef PAPYTELEMAPP_H_INCLUDED

